<?php

namespace App\Http\Controllers\AppHandler\User;

use App\Http\Controllers\Controller;
use App\Models\AUser;
use App\Models\DB_Daily;
use App\Models\DB_Refer;
use App\Models\DB_Settings;
use App\Models\DB_Tracking;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class Signup extends Controller
{
    public static function register(Request $request)
    {
        try {
            // Extract request data
            $android_id   = $request->input('android_id');
            $u_id         = $request->input('u_id');
            $name         = $request->input('name');
            $email        = $request->input('email');
            $country_name = $request->input('country_name');
            $refer        = $request->input('refer');
            $referred_by  = $request->input('referred_by'); // can be null
            $image        = $request->input('image');
            $f_token      = $request->input('f_token');

            // Fetch signup bonus
            $app_config = json_decode(DB_Settings::where('id', 1)->value('app_config'), true);
            $signup_bonus = (int)($app_config['signup_bonus'] ?? 0);

            // Check if user already exists
            $existingUser = AUser::where('email', $email)->first();
            if ($existingUser) {
                $existingUser->update([
                    'android_id' => $android_id,
                    'fcmtoken'   => $f_token,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Welcome Back!',
                    'user'    => [$existingUser]
                ], 200);
            }

            // Validate referral code if provided
            if (!empty($referred_by)) {
                $refer_check = AUser::where('refer', $referred_by)->first();
                if (!$refer_check) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid referral code!'
                    ], 201);
                }
            }

            // Get referral points from settings
            $refer_config_json = DB_Settings::where('id', 1)->value('refer_config');
            $refer_config = json_decode($refer_config_json, true);
            $level1_point = isset($refer_config['level1_point']) ? (int)$refer_config['level1_point'] : 0;
            $level2_point = isset($refer_config['level2_point']) ? (int)$refer_config['level2_point'] : 0;
            $pointLevels = [$level1_point, $level2_point];
            $maxLevels = count($pointLevels);

            // Create new user safely
            $userData = [
                'name'         => $name,
                'email'        => $email,
                'u_id'         => $u_id,
                'country_name' => $country_name,
                'android_id'   => $android_id,
                'password'     => Hash::make('123456'),
                'point'         => $signup_bonus,
                'refer'        => $refer,
                'signup_type'  => 'Google',
                'fcmtoken'     => $f_token,
                'image'        => $image
            ];

            // Only include referred_by if it exists
            if (!empty($referred_by)) {
                $userData['referred_by'] = $referred_by;
            }

            $user = AUser::create($userData);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Account creation unsuccessful'
                ], 404);
            }

            // Track signup bonus
            DB_Tracking::create([
                'u_id'   => $u_id,
                'title'  => 'Signup Bonus Reward',
                'reward' => $signup_bonus
            ]);

            // Create daily record
            DB_Daily::create(['u_id' => $u_id]);

            // Process referrals (only if user signed up with referral)
            if (!empty($referred_by)) {
                $level = 0;
                $currentUser = $user;

                while ($currentUser && !empty($currentUser->referred_by) && $level < $maxLevels) {
                    $referrer = AUser::where('refer', $currentUser->referred_by)->first();
                    if (!$referrer) break;

                    $referralPoints = $pointLevels[$level];

                    DB_Refer::create([
                        'u_id'  => $referrer->u_id,
                        'r_uid' => $currentUser->u_id,
                        'level' => $level + 1,
                        'point' => $referralPoints
                    ]);

                    // Add referral points to referrer
                    $referrer->increment('point', $referralPoints);

                    $currentUser = $referrer;
                    $level++;
                }

                // Update total first-level referrals
                $referrerUser = AUser::where('refer', $referred_by)->first();
                if ($referrerUser) {
                    $referrerUser->total_refer = DB_Refer::where('level', 1)
                        ->where('u_id', $referrerUser->u_id)
                        ->count();
                    $referrerUser->save();
                }
            }
            
            $settings = DB_Settings::find(1);
            return response()->json([
                'success' => true,
                'message' => 'User registration successful!',
                'user'    => [$user],
                'setting' => $settings,
            ], 200);

        } catch (\Exception $e) {
            Log::error('Signup Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing your request.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
}
